<?php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\Membership;
use App\Models\Transaction;
use App\Models\Attendance;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;

class ReportsController extends Controller
{
    /**
     * Display information about members based on search criteria.
     *
     * @param  \Illuminate\Http\Request  $request  The incoming request.
     * @return \Illuminate\Contracts\View\View     The view displaying member information.
     */
    public function member_information(Request $request)
    {
        if ($request->has('search')) {
            $keyword = $request->input('search');
        } else {
            $keyword = Session::get('keyword');
        }

        if (empty($keyword)) {
            $keyword = "all";
        }

        Session::put('keyword', $keyword);

        $type = $request->query('type');
        $today = Carbon::now()->startOfDay();
        $one_week_later = $today->copy()->addWeek();

        $paginate = true;
        $query = DB::table('members')
            ->join('memberships', 'members.id', '=', 'memberships.member_id')
            ->join('membership_types', 'memberships.membership_type_id', '=', 'membership_types.id')
            ->leftJoin('transactions', 'memberships.id', '=', 'transactions.transactionable_id')
            ->where('memberships.created_at', function ($query) {
                $query->select('created_at')
                    ->from('memberships')
                    ->whereColumn('member_id', 'members.id')
                    ->orderByDesc('created_at')
                    ->limit(1);
            })
            ->select(
                'members.id AS member_id',
                'members.reg_no',
                'members.name',
                'members.phone',
                'memberships.valid_from',
                'memberships.valid_to',
                'memberships.id',
                'membership_types.name AS membership_name',
                'membership_types.amount AS actual_amount',
                DB::raw('COALESCE(SUM(transactions.amount), 0) AS total_transactions'),
                DB::raw('membership_types.amount - COALESCE(SUM(transactions.amount), 0) AS due_amount')
            )
            ->groupBy(
                'members.id',
                'members.reg_no',
                'members.name',
                'members.phone',
                'memberships.valid_from',
                'memberships.valid_to',
                'memberships.id',
                'membership_types.name',
                'membership_types.amount',
            )
            ->orderByDesc('memberships.valid_to');

        if ($keyword == "all") {
            if ($type == "print" || $type == "csv") {
                $members = $query->get();
            } else {
                $members = $query->paginate(10)->onEachSide(0);
            }
        } else if ($keyword == "expired") {
            if ($type == "print" || $type == "csv") {
                $members = $query->where('memberships.valid_to', '<', $today)->get();
            } else {
                $members = $query->where('memberships.valid_to', '<', $today)->paginate(10)->onEachSide(0);
            }
        } else if ($keyword == "near_expiry") {
            if ($type == "print" || $type == "csv") {
                $members = $query->whereBetween('memberships.valid_to', [$today, $one_week_later])->get();
            } else {
                $members = $query->whereBetween('memberships.valid_to', [$today, $one_week_later])->paginate(10)->onEachSide(0);
            }
        } else {
            if ($type == "print" || $type == "csv") {
                $members = $query->where('memberships.valid_to', '>', $today)->get();
            } else {
                $members = $query->where('memberships.valid_to', '>', $today)->paginate(10)->onEachSide(0);
            }
        }

        if ($type == "print") {
            return view('reports.print.member_information', compact('members'));
        } else if ($type == "csv") {
            $this->generate_csv($members, "memberships");
        } else {
            return view('reports.member_information', compact('members', 'paginate', 'keyword'));
        }
    }

    public function payment_transactions(Request $request)
    {
        if ($request->has('search')) {
            $keyword = $request->input('search');
        } else {
            $keyword = Session::get('keyword');
        }

        if (empty($keyword)) {
            $keyword = "all";
        }

        Session::put('keyword', $keyword);

        $type = $request->query('type');
        $paginate = true;
        $dates = getDates($keyword);
        $query = Transaction::with('transactionable');

        if ($keyword != "all" && $dates[0] != NULL && $dates[1] != NULL) {
            $date1 = $dates[0];
            $date2 = $dates[1];
            if ($type == "print" || $type == "csv") {
                $payments = $query->whereBetween('created_at', [$date1, $date2])->orderByDesc('created_at')->get();
            } else {
                $payments = $query->whereBetween('created_at', [$date1, $date2])->orderByDesc('created_at')->paginate(10)->onEachSide(0);
            }

        } else {
            if ($type == "print" || $type == "csv") {
                $payments = $query->orderByDesc('created_at')->get();
            } else {
                $payments = $query->orderByDesc('created_at')->paginate(10)->onEachSide(0);
            }
        }

        if ($type == "print") {
            return view('reports.print.payment_transaction', compact('payments'));
        } else if ($type == "csv") {
            $this->generate_csv($payments, "payments");
        } else {
            return view('reports.payment_transaction', compact('payments', 'paginate', 'keyword'));
        }
    }

    /**
     * Display payment transactions based on search criteria.
     *
     * @param  \Illuminate\Http\Request  $request  The incoming request.
     * @return \Illuminate\Contracts\View\View     The view displaying payment transactions.
     */
    public function attendance(Request $request)
    {
        if ($request->has('search')) {
            $keyword = $request->input('search');
        } else {
            $keyword = Session::get('keyword');
        }

        if (empty($keyword)) {
            $keyword = "all";
        }

        Session::put('keyword', $keyword);

        if ($request->has('member')) {
            $member = $request->input('member');
        } else {
            $member = Session::get('member');
        }

        if (empty($member)) {
            $member = "all";
        }

        Session::put('member', $member);

        $members = Member::where('status', 'active')->get();
        $type = $request->query('type');
        $paginate = true;
        $dates = getDates($keyword);
        $date1 = $dates[0];
        $date2 = $dates[1];

        if ($keyword != "all" && $dates[0] != NULL && $dates[1] != NULL) {
            $query = Attendance::with('member', 'user')->whereBetween('attendance_date', [$date1, $date2]);

            if (empty($member) || $member == "all") {
                if ($type == "print" || $type == "csv") {
                    $attendances = $query->orderByDesc('attendance_date')->get();
                } else {
                    $attendances = $query->orderByDesc('attendance_date')->paginate(10)->onEachSide(0);
                }
            } else {
                if ($type == "print" || $type == "csv") {
                    $attendances = $query->where('member_id', $member)->orderByDesc('attendance_date')->get();
                } else {
                    $attendances = $query->where('member_id', $member)->orderByDesc('attendance_date')->paginate(10)->onEachSide(0);
                }
            }

        } else {
            $query = Attendance::with('member', 'user');
            if (empty($member) || $member == "all") {
                if ($type == "print" || $type == "csv") {
                    $attendances = $query->orderByDesc('attendance_date')->get();
                } else {
                    $attendances = $query->orderByDesc('attendance_date')->paginate(10)->onEachSide(0);
                }
            } else {
                if ($type == "print" || $type == "csv") {
                    $attendances = $query->where('member_id', $member)->orderByDesc('attendance_date')->get();
                } else {
                    $attendances = $query->where('member_id', $member)->orderByDesc('attendance_date')->paginate(10)->onEachSide(0);
                }
            }
        }
      
        if ($type == "print") {
            return view('reports.print.attendance', compact('attendances'));
        } else if ($type == "csv") {
            $this->generate_csv($attendances, "attendance");
        } else {
            return view('reports.attendance', compact('attendances', 'paginate', 'keyword', 'members', 'member'));
        }
    }

    /**
     * Generate a CSV file for given collections based on type.
     *
     * @param  mixed  $collections  The collections to be included in the CSV.
     * @param  string $type         The type of data being exported.
     * @return void
     */
    private function generate_csv($collections, $type)
    {
        $file_name = $type . date("Ymd");
        header("Content-Description: File Transfer");
        header("Content-Disposition: attachment; filename=$file_name.csv");
        header("Content-Type: application/csv; ");

        // file creation 
        $file = fopen('php://output', 'w');
        $csv_data = [];

        if ($type == "memberships") {
            $header = [
                __('reports.th_reg_no'),
                __('reports.th_member_name'),
                __('reports.th_reg_no'),
                __('reports.th_membership_type'),
                __('reports.th_valid_from'),
                __('reports.th_valid_to'),
                __('reports.th_status'),
            ];

            foreach ($collections as $collection) {
                $csv_data[] = [
                    $collection->reg_no,
                    $collection->name,
                    $collection->phone,
                    $collection->membership_name,
                    $collection->valid_from,
                    $collection->valid_to,
                    membership_status_text($collection->valid_to),
                ];
            }
        } else if ($type == "payments") {
            $header = [
                __('reports.th_name'),
                __('reports.th_amount'),
                __('reports.th_date'),
                __('reports.th_transaction_type'),
                __('reports.th_payment_method'),
                __('reports.th_details'),
            ];

            foreach ($collections as $collection) {
                $amount = 0;

                if ($collection->dr_cr == 'cr') {
                    $amount = '+' . $collection->amount;
                } else {
                    $amount = '-' . $collection->amount;
                }

                $csv_data[] = [
                    transaction_name($collection),
                    $amount,
                    $collection->date,
                    $collection->type,
                    $collection->method,
                    $collection->note,
                ];
            }
        } else if ($type == "attendance") {
            $header = [
                __('reports.th_reg_no'),
                __('reports.th_name'),
                __('reports.th_date'),
                __('reports.th_day'),
                __('reports.th_status'),
                __('reports.th_taken_by'),
            ];

            foreach ($collections as $collection) {
                $csv_data[] = [
                    $collection->member->reg_no,
                    $collection->member->name,
                    $collection->attendance_date,
                    day_name($collection->attendance_date),
                    $collection->status,
                    $collection->user->name,
                ];
            }
        } else {
            return back()->with("error", "Invalid Request");
        }

        fputcsv($file, $header);
        foreach ($csv_data as $row) {
            fputcsv($file, $row);
        }
        fclose($file);
        exit;
    }
}